' Sunset puzzle as seen on YouTube channel "Mr Puzzle"
' Rev 1.0.0 William M Leue 5/16/2021
' Rev 1.1.0 5/25/2021 - added a solution (there are others)
option default integer
option base 1

' Constants
const USIZE = 60
const FWSPACE = 4
const FHSPACE = 5
const FTHICK = 8
const NUM_PIECES = 10
const NUM_ROWS = 5
const NUM_COLS = 4
const DRAD = 0.7*USIZE\2
const NUM_COLORS = 3
const FCOLOR = RGB(80, 30, 10)
const PBCOLOR = RGB(100, 80, 50)

const UP = 128
const DOWN = 129
const LEFT = 130
const RIGHT = 131
const SPACE = 32
const ESCAPE = 27
const HOME = 134
const ATSGN = 64
const ENTER = 13

' Globals
dim cx, cy
dim pieces(NUM_PIECES, 5)
dim board(NUM_ROWS, NUM_COLS)
dim colors(NUM_COLORS) = (RGB(BLACK), RGB(RED), RGB(YELLOW))
dim selected = 0
dim captured = 0
dim nmoves = 0
dim solution(500, 2)
dim nwinmoves = 0
dim winspeed = 0
dim nsunmoves(6) = (38, 134, 156, 166, 174, 182)
dim nsunnames$(6) = ("1 time", "2 times", "3 times", "4 times", "5 times", "Win!")
dim nmovespersec(4) = (1, 2, 4, 8)
dim pmode = 1

' Main Program
open "debug.txt" for output as #1
ReadPieces
ReadMoves
ShowRules
selected = 1
DrawPuzzle
HandleUserEvents
end

' Read the pieces, their initial location, and spot colors
' Also init the board squares
sub ReadPieces
  local i, j, row, col, sr, er, sc, ec
  for row = 1 to NUM_ROWS
    for col = 1 to NUM_COLS
      board(row, col) = 0
    next col
  next row
  for i = 1 to NUM_PIECES
    for j = 1 to 5
      read pieces(i, j)
    next j
    sr = pieces(i, 1) : er = sr + pieces(i, 4)-1
    sc = pieces(i, 2) : ec = sc + pieces(i, 3)-1
    for row = sr to er
      for col = sc to ec
        board(row, col) = i
      next col
    next row
  next i
end sub

' read moves
sub ReadMoves
  local i, j
  read nwinmoves
  for i = 1 to nwinmoves
    for j = 1 to 2
      read solution(i, j)
    next j
  next i
end sub

' Reset the puzzle to the initial position
sub NewGame
  cls
  Restore
  ReadPieces
  DrawPuzzle
  nmoves = 0
  selected = 1
end sub

' Draw the Puzzle
sub DrawPuzzle
  page write 1
  DrawFrame
  if pmode = 1 then
    DrawKeyMap
  else
    ClearKeyMap
    DrawSolutionText
  end if
  DrawPieces 1
  page write 0
  page copy 1 to 0, B
end sub

' Draw the Frame of the Puzzle
sub DrawFrame
  local x, y, x1, y1, x2, y2
  text MM.HRES\2, 10, "The Sunset Puzzle", "CT", 5
  cx = MM.HRES\2 : cy = MM.VRES\2 - 40
  x = cx - 2*USIZE
  y = cy + USIZE\2 - 2.5*USIZE
  box x-FTHICK, y-FTHICK, FWSPACE*USIZE+2*FTHICK, FHSPACE*USIZE+2*FTHICK, FTHICK, FCOLOR, FCOLOR
  x1 = cx-USIZE : y1 = cy+3*USIZE+2 : x2 = cx+USIZE : y2 = y1
  line x1, y1, x2, y2, 3, RGB(BLUE)
end sub

' Draw a reminder of the keyboard commands
sub DrawKeyMap
  text MM.HRES\2, 530, "Arrow Keys to Navigate and Move Pieces", "CT"
  text MM.HRES\2, 545, "Spacebar to Capture or Free a Piece", "CT"
  text MM.HRES\2, 560, "Home key to restart the Puzzle", "CT"
  text MM.HRES\2, 575, "Escape key to Quit", "CT"
end sub

' Clear the key map text
sub ClearKeyMap
  box 0, 530, 799, 599,, RGB(BLACK), RGB(BLACK)
end sub

' Draw some text about the automated solution
sub DrawSolutionText
  local spd$, nmv$, i
  if nwinmoves < 6 then
    text MM.HRES\2, 530, "The program is showing a partial automated solution", "CT"
  else
    text MM.HRES\2, 530, "The program is showing a full automated solution", "CT"
  end if
  spd$ = "at " + str$(winspeed) + " moves per second."
  text MM.HRES\2, 545, spd$, "CT"
  nmv$ = "This requires " + str$(nsunmoves(nwinmoves)) + " moves."
  text MM.HRES\2, 560, nmv$, "CT"
end sub

' Draw the pieces whatever their current location
' if label <> 0, label the pieces with their numbers
sub DrawPieces label
  local i, row, col, w, h, x, y, bx, by, c, ec
  for i = 1 to NUM_PIECES
    row = pieces(i, 1) : col = pieces(i, 2)
    w = pieces(i, 3)   : h = pieces(i, 4)
    c = pieces(i, 5)
    if i = selected then
      if captured then
        ec = RGB(GREEN)
      else
        ec = RGB(WHITE)
      end if
    else
      ec = RGB(BLACK)
    end if
    x = cx - 2*USIZE + (col-1)*USIZE
    y = cy - 2*USIZE + (row-1)*USIZE
    box x, y, w*USIZE, h*USIZE,, ec, PBCOLOR
    bx = x + w*USIZE\2
    by = y + h*USIZE\2
    circle bx, by, DRAD,,, colors(c), colors(c)
    if label then
      text bx, by, str$(i), "CM", 7,, RGB(GREEN), -1
    end if
  next i    
end sub

' Handle all user interaction when the puzzle is being solved
sub HandleUserEvents
  local z$
  local cmd, row, col, p, move, old_row, old_col
  row = 1 : col = 1
  z$ = INKEY$
  p = selected
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    move = 0
    cmd = asc(z$)
    old_row = row : old_col = col
    select case cmd
      case UP
        if row > 1 then
          row = row-1
        else
          row = NUM_ROWS
        end if
        move = 1
      case DOWN
        if row < NUM_ROWS then
          row = row+1
        else
          row = 1
        end if
        move = 1
      case LEFT
        if col > 1 then
          col = col-1
        else
          col = NUM_COLS
        end if
        move = 1
      case RIGHT
        if col < NUM_COLS then
          col = col+1
        else
          col = 1
        end if
        move = 1
      case SPACE
        captured = 1 - captured
        move = 0
      case HOME
        NewGame
      case ATSGN
        HandleSolutionEvents
      case ESCAPE
        cls
        end
    end select
    if move then
      if captured then
        MaybeMovePiece p, cmd
      else
        p = NextPieceInDirection(p, cmd)
        selected = p
      end if
    end if
    DrawPuzzle
  loop
end sub

' Given the captured piece index and the direction cmd, try to move the piece
sub MaybeMovePiece p, cmd
  local prow, pcol, pw, ph
  if p = 0 then exit sub
  prow = pieces(p, 1) : pcol = pieces(p, 2)
  pw = pieces(p, 3)   : ph = pieces(p, 4)
  select case cmd
    case UP, DOWN
      if HasVMoveSpace(p, cmd) then
        VMovePiece p, cmd
        CheckWin
      end if
    case LEFT, RIGHT
      if HasHMoveSpace(P, cmd) then
        HMovePiece p, cmd
        CheckWin
      end if
  end select
end sub

' Check for space above or below a piece that
' the piece can move into. Returns 1 for yes
' and 0 for no.
' Parameters: p: piece index, d: direction command
function HasVMoveSpace(p, d)
  local ok, prow, pcol, pw, ph, row, col
  prow = pieces(p, 1) : pcol = pieces(p, 2)
  pw = pieces(p, 3) : ph = pieces(p, 4)
  ok = 1
  if d = UP then
    row = prow-1
  else
    row = prow+ph
  end if
  if row < 1 or row > NUM_ROWS then
    HasVMoveSpace = 0
    exit function
  end if
  for col = pcol to pcol+pw-1
    if board(row, col) <> 0 then ok = 0
  next col
  HasVMoveSpace = ok
end function

' Check for space left or right of a piece that
' the piece can move into. Returns 1 for yes
' and 0 for no.
' Parameters: p: piece index, d: direction command
function HasHMoveSpace(p, d)
  local prow, pcol, pw, ph, row, col
  prow = pieces(p, 1) : pcol = pieces(p, 2)
  pw = pieces(p, 3)   : ph = pieces(p, 4)
  ok = 1
  if d = LEFT then
    col = pcol-1
    if col < 1 or col+pw-1 > NUM_COLS then
      HasHMoveSpace = 0
      exit function
    end if
  else
    col = pcol+pw
    if col < 1 or col > NUM_COLS then
      HasHMoveSpace = 0
      exit function
    end if
    limcol = col+pw-1
  end if
  for row = prow to prow+ph-1  
    if board(row, col) <> 0 then ok = 0
  next row
  HasHMoveSpace = ok
end function

' Unconditionally move a piece in the vertical
' dimension and update the board.
sub VMovePiece p, cmd
  local col, prow, pcol, pw, ph
  prow = pieces(p, 1) : pcol = pieces(p, 2)
  pw = pieces(p, 3)   : ph = pieces(p, 4)
  select case cmd
    case UP
      pieces(p, 1) = prow-1
      for col = pcol to pcol+pw-1
        board(prow-1, col) = p
        board(prow+ph-1, col) = 0
      next col
    case DOWN
      pieces(p, 1) = prow+1
      for col = pcol to pcol+pw-1
        board(prow+ph, col) = p
        board(prow, col) = 0
      next col
  end select
end sub

' Unconditionally move a piece in the horizontal
' dimension and update the board.
sub HMovePiece p, cmd
  local prow, pcol, pw, ph, row, col
  prow = pieces(p, 1) : pcol = pieces(p, 2)
  pw = pieces(p, 3)   : ph = pieces(p, 4)
  select case cmd
    case LEFT
      pieces(p, 2) = pcol-1
      for row = prow to prow+ph-1
        board(row, pcol-1) = p
        board(row, pcol+pw-1) = 0
      next row
    case RIGHT
      pieces(p, 2) = pcol+1
      for row = prow to prow+ph-1
        board(row, pcol+pw) = p
        board(row, pcol) = 0
      next row
  end select
end sub

' Find the next piece in the indicated direction from the current piece,
' and return its index. If no piece in that direction, wrap around to
' start and find the first piece, which might be the same as the current
' piece. P is the current piece index, d is the direction command.
function NextPieceInDirection(p, d)
  local prow, pcol, pw, ph, crow, ccol, rstart, cstart, b, np, hit
  prow = pieces(p, 1) : pcol = pieces(p, 2)
  pw = pieces(p, 3) : ph = pieces(p, 4)
  np = p
  hit = 0
  select case d
    case UP
      if prow > 1 then
        rstart = prow-1
      else
        rstart = NUM_ROWS
      end if
      for crow = rstart to 1 step -1
        for ccol = pcol to pcol+pw-1
          b = board(crow, pcol)
          if b > 0 and b <> p then
            np = b
            hit = 1
            exit for
          end if
        next ccol
        if hit then exit for
      next crow
    case DOWN
      if prow < NUM_ROWS then
        rstart = prow+1
      else
        rstart = 1
      end if
      np = p
      for crow = rstart to NUM_ROWS
        for ccol = pcol to pcol+pw-1
          b = board(crow, ccol)
          if b > 0 and b <> p then
            np = b
            hit = 1
            exit for
          end if
        next ccol
        if hit then exit for
      next crow
    case LEFT
      if pcol > 1 then
        cstart = pcol-1
      else
        cstart = NUM_COLS
      end if
      for ccol = cstart to 1 step -1
        for crow = prow to prow+ph-1
          b = board(crow, ccol)
          if b > 0 and b <> p then
            np = b
            hit = 1
            exit for
          end if
        next crow
        if hit then exit for
      next ccol
    case RIGHT
      if pcol < NUM_COLS then
        cstart = pcol+1
      else
        cstart = 1
      end if
      for ccol = cstart to NUM_COLS
        for crow = prow to prow+ph-1
          b = board(crow, ccol)
          if b > 0 and b <> p then
            np = b
            hit = 1
            exit for
          end if
        next crow
        if hit then exit for
      next ccol
  end select
  NextPieceInDirection = np
end function

' Clear piece selection
sub ClearSelection
  selected = 0
  captured = 0
end sub

' Check for a Win, track number of moves, and notify user
sub CheckWin
  local win = 0
  inc nmoves
  text 600, 30, space$(22)
  text 600, 200, "Number of Moves: " + str$(nmoves)
  if board(5, 2) = 2 and board(5, 3) = 2 then
    win = 1
  end if
  if win then
    text MM.HRES\2, 570, "YOU WIN!!", "CB", 5, 2, RGB(RED)
    text MM.HRES\2, 599, "Press any key to Quit", "CB"
    z$ = INKEY$
    do
      z$ = INKEY$
    loop until z$ <> ""
    cls
    end
  end if
end sub

sub ShowRules
  cls
  text MM.HRES\2, 10, "The Sunset Puzzle", "CT", 5
  text 30, 60, "Want to See Instructions? (Y, N)"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  if LEFT$(UCASE$(z$), 1) <> "Y" then
    cls
    exit sub
  end if
  text 0, 80, ""
  print "The Sunset Puzzle is a sliding block puzzle similar to the classic '15' puzzle."
  print "The goal is to get the 'Sun' piece (the 2x2 block with the red circle) to the"
  print "bottom center of the board. On a real puzzle, there is a slot on the bottom side"
  print "that is wide enough to admit the Sun piece but none of the others, which are"
  print "thicker. (This game shows the winning location for the bottom of the Sun piece with a"
  print "blue line.) Pieces can only be moved left, right, up, and down and only when there"
  print "is a space to accomodate them. No piece can be rotated or flipped."
  print "In the optimum solution it still takes more than 100 moves to solve."
  print ""
  print "Use the arrow keys to navigate around the board. Your position is marked by a white"
  print "outline on the current block. However, you first have to 'capture' the marked block"
  print "by pressing the space bar before you can move it. A 'captured' block is marked by"
  print "a green outline. Once you have captured a block, use the arrow keys to move it to"
  print "a suitable empty space on the board. Press space again to release a captured block."
  print ""
  print "Press the Home key to restart the game."
  print "You can also press the Escape key any time to quit."
  print ""
  print "The Sunset Puzzle was shown and solved on YouTube on the 'Mr Puzzle' channel. He shows"
  print "a solution if you get stuck and need help."
  print ""
  print "You can also press the '@' key to see a partial or full solution animated by the"
  print "program. I suggest you try one of the partial solutions (e.g. sun piece moves 1 time)"
  print "instead of going for the full solution the first time. The full solution shown takes"
  print "182 steps, although shorter solutions are possible."

  text MM.HRES\2, 550, "Press any key to start", "CT", 3
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$  <> ""
  cls
end sub

' Handle User Inputs for showing full or partial solution
sub HandleSolutionEvents
  local cmd, active, nsteps, nmps
  z$ = INKEY$
  active = 1
  nsteps = 1
  nmps = 1
  DrawSolutionScreen nsteps, nmps
  HiliteSolutionScreen active
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(z$)
    select case cmd
      case UP
        if active > 1 then
          active = active-1
        else
          active = 2  
        end if
      case DOWN
        if active < 2 then
          active = active+1
        else
          active = 1
        end if
      case LEFT
        if active = 1 then
          if nsteps > 1 then
            nsteps = nsteps-1
          else
            nsteps = 6
          end if
        else
          if nmps > 1 then
            nmps = nmps-1
          else
            nmps = 4
          end if
        end if
      case RIGHT
        if active = 1 then
          if nsteps < 6 then
            nsteps = nsteps+1
          else
            nsteps = 1
          end if
        else
          if nmps < 4 then
            nmps = nmps+1
          else
            nmps = 1
          end if
        end if
      case ENTER
        nwinmoves = nsteps
        winspeed = nmovespersec(nmps)
        ShowSolution nwinmoves, winspeed
      case ESCAPE
        close #1
        cls
        end
    end select
    pmode = 2
    DrawSolutionScreen nsteps, nmps
    HiliteSolutionScreen active
  loop
end sub

' Hilight the Active Element on the Solution screen
sub HiliteSolutionScreen which
  local xv(3), yv(3), y
  static oy = 0
  y = 213+(which-1)*100
  xv(1) = 48
  xv(2) = 38
  xv(3) = 38
  if oy > 0 then
    yv(1) = oy
    yv(1) = oy-5
    yv(3) = oy+5
    polygon 3, xv(), yv(), RGB(BLACK), RGB(BLACK)
  end if
  yv(1) = y
  yv(2) = y-5
  yv(3) = y+5
  polygon 3, xv(), yv(), RGB(RED), RGB(RED)
  oy = y
end sub

' Screen to let user choose number of solution steps to show
' and how fast to display the steps.
sub DrawSolutionScreen nsteps, nmps
  cls
  text MM.HRES\2, 10, "Sunset Puzzle Solution", "CT", 5
  text MM.HRES\2, 45, "Use UP, DOWN arrow keys to select a parameter", "CT"
  text MM.HRES\2, 60, "Use LEFT, RIGHT arrow keys to select a value", "CT"
  text MM.HRES\2, 75, "Press ENTER to lock in values and run solution", "CT"
  text MM.HRES\2, 90, "Press ESCAPE to quit.", "CT"
  text 50, 200, "Show Steps Until Sun Moves...", "LT", 3
  text 80, 230, nsunnames$(nsteps),,,, RGB(CYAN)
  text 50, 300, "Number of Moves Per Second", "LT", 3
  text 80, 330, str$(nmovespersec(nmps)),,,, RGB(CYAN)
end sub

' show the solution
sub ShowSolution nsteps, winspeed
  local i, n, ticks
  NewGame
  ClearSelection
  nwinmoves = nsteps
  for i = 1 to nsunmoves(nwinmoves)
    DoMove(solution(i, 1), solution(i, 2))
    DrawPuzzle
    CheckWin
    pause 1000\winspeed
  next i
end sub

' make a move under computer control
sub DoMove piece, dir
  local cmd
  select case dir
    case 1 : cmd = UP
    case 2 : cmd = DOWN
    case 3 : cmd = LEFT
    case 4 : cmd = RIGHT
  end select
  if cmd = UP or cmd = DOWN then
    VMovePiece piece, cmd
  else
    HMovePiece piece, cmd
  end if
end sub

' Initial Pieces (row, col, width, height, color)
' Note that the 1x2 pieces come in 4 vertical and 1 horizontal orientation,
' but neither can be rotated. The other pieces are symmetrical.
data 1, 1, 1, 2, 1
data 1, 2, 2, 2, 2
data 1, 4, 1, 2, 1
data 3, 2, 2, 1, 1
data 4, 1, 1, 2, 1
data 4, 2, 1, 1, 3
data 4, 3, 1, 1, 3
data 4, 4, 1, 2, 1
data 5, 2, 1, 1, 3
data 5, 3, 1, 1, 3

' Secret Sauce for a Solution (not necessarily the fewest moves)
' Each move is (piece, direction) 1=UP,2=DOWN,3=LEFT,4=RIGHT
data 182
data 4,4,  6,1,  6,3,  9,1,  9,1,  5,4,  6,2,  6,2,  9,3,  9,2,  4,3,  4,3
data 7,1,  7,4,  10,1, 10,1, 5,4,  9,4,  9,2,  4,2,  10,3, 10,3, 7,3,  7,3
data 5,1,  8,1,  9,4,  9,4,  6,4,  6,4,  4,2,  7,2,  7,3,  5,3,  8,3,  3,2 
data 3,2,  2,4,  5,1,  5,1,  8,3,  3,3,  9,1,  9,1,  6,4,  6,1,  4,4,  4,4
data 8,2, 10,4,  7,1,  8,3,  10,2, 10,2, 7,4,  7,2,  8,1,  10,3, 7,2,  3,3
data 9,3,  6,1,  4,1,  7,4,  7,4,  10,4, 10,4, 8,2,  3,2,  9,3,  9,3,  6,3
data 6,3,  4,1,  7,1,  10,4, 3,4,  6,2,  6,2,  9,4,  9,2,  8,1,  6,3,  9,2
data 5,2,  5,2,  1,4,  8,1,  8,1,  6,1,  6,1,  9,3,  9,1,  5,2,  6,4,  9,1
data 5,3,  6,2,  6,2,  1,2,  1,2,  8,4,  9,1,  9,1,  5,1,  5,1,  6,3,  6,1
data 1,2,  8,2,  9,4,  5,1,  6,1,  1,3,  8,2,  8,2,  9,2,  9,2,  5,4,  6,1
data 6,1,  9,3,  9,1,  5,2,  6,4,  9,1,  1,1,  1,1,  8,3,  3,3, 10,3,  7,2
data 4,2,  2,2,  6,4,  6,4,  9,4,  9,4,  5,1,  1,1,  8,1,  3,1, 10,3, 10,3
data 7,3,  7,3,  4,2,  2,2,  9,2,  9,4,  5,4,  1,4,  8,1,  8,1,  3,3,  2,3
data 9,2,  9,2,  6,2,  6,2,  5,4,  1,4,  8,4,  3,1,  3,1,  2,3,  6,3,  9,1
data 4,1,  7,4,  7,4, 10,4, 10,4,  2,2,  6,3,  6,3,  9,3,  9,3,  4,1, 10,1
data 10,4, 2,4
